<?php

namespace App\Services;

use App\Models\ConfigDatabase;
use App\Models\Product;
use App\Services\Odoo\OdooApiInterface;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;

class ProductService
{
    public function __construct(private readonly OdooApiInterface $odoo) {}

    public function syncAllServers(): int
    {
        $user = Auth::user();
        if (! $user) {
            throw new \RuntimeException('Access denied');
        }

        $servers = ConfigDatabase::query()
            ->whereNotNull('client_id')
            ->where('client_id', '!=', '')
            ->get();

        $count = 0;

        foreach ($servers as $server) {
            $token = $this->odoo->getToken($user->email, (string) $user->odoo_password, $server);

            $parameters = [
                'body' => http_build_query([
                    'db'      => $server->database,
                    'domain'  => '[["sale_ok","=",true]]',
                    'fields'  => '["id","name","list_price","taxes_id","gg_business_unit_id"]',
                    'limit'   => 0,
                ]),
                'headers' => ['Content-Type' => 'application/x-www-form-urlencoded'],
            ];

            $resp = $this->odoo->searchRead('product.product', $token->access_token, $parameters, $server);

            if (($resp['status_code'] ?? 500) >= 400 || empty($resp['data'])) {
                continue;
            }

            $rows = collect($resp['data']);

            $count += $this->upsertProducts($rows, $server->id);
        }

        return $count;
    }

    private function upsertProducts(Collection $rows, int $serverId): int
    {
        $updated = 0;

        foreach ($rows as $row) {
            // business unit
            $businessUnit = 0;
            if (!empty($row->gg_business_unit_id) && is_array($row->gg_business_unit_id)) {
                $businessUnit = (int) ($row->gg_business_unit_id[0] ?? 0);
            }

            // taxes flag
            $taxesFlag = 0;
            if (!empty($row->taxes_id) && is_array($row->taxes_id)) {
                foreach ($row->taxes_id as $tax) {
                    $taxesFlag = ($tax == 2) ? 1 : 0;
                }
            }

            $match = [
                'odoo_product_id' => (int) $row->id,
                'odoo_server'     => $serverId,
            ];

            $attrs = [
                'name'             => (string) ($row->name ?? ''),
                'list_price'       => (float) ($row->list_price ?? 0),
                'taxes'            => $taxesFlag,
                'business_unit_id' => $businessUnit,
            ];

            $product = Product::updateOrCreate($match, $attrs);
            $product->save();
            $updated++;
        }

        return $updated;
    }
}
