<?php

namespace App\Services\Odoo;

use AllowDynamicProperties;
use App\Models\ConfigDatabase;
use App\Models\UserToken;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Auth;
use League\OAuth2\Client\Provider\GenericProvider;

#[AllowDynamicProperties]
class OdooApiService implements OdooApiInterface
{
    public function __construct(
        ?string $apiHost = null,
        private readonly Client $http = new Client(['verify' => false]),
    ) {
        $this->apiHost = $apiHost ?: (string) config('services.odoo.host', 'erp.glaucoma.mx:8069');
    }

    public function getProvider(ConfigDatabase $server): GenericProvider
    {
        return new GenericProvider([
            'clientId'                => $server->client_id,
            'clientSecret'            => $server->client_secret,
            'redirectUri'             => 'http://app.swaggerhub.com/oauth2_redirect/',
            'urlAuthorize'            => 'http://' . $this->apiHost . '/api/authentication/oauth2/authorize',
            'urlAccessToken'          => 'http://' . $this->apiHost . '/api/authentication/oauth2/token',
            'urlResourceOwnerDetails' => '',
        ]);
    }

    public function getToken(string $username, string $password, ConfigDatabase $server): UserToken
    {
        $existing = UserToken::where('user_id', Auth::id())
            ->where('server_id', $server->id)
            ->where('expires_at', '>', Carbon::now())
            ->first();

        if ($existing) {
            return $existing;
        }

        $provider = $this->getProvider($server);

        $credentials = [
            'username' => $username,
            'password' => $password,
            'db' => $server->database
        ];

        $accessToken = $provider->getAccessToken('password', $credentials);

        $token = new UserToken();
        $token->user_id = Auth::id();
        $token->server_id = $server->id;
        $token->access_token = $accessToken->getToken();
        $token->refreshToken = $accessToken->getRefreshToken();
        $token->expires_at = Carbon::now()->addMinutes(55);
        $token->save();

        return $token;
    }

    public function userInfo(string $token, array $options, ConfigDatabase $server): array
    {
        try {
            $provider = $this->getProvider($server);
            $request = $provider->getAuthenticatedRequest(
                'GET',
                'http://' . $this->apiHost . '/api/userinfo',
                $token,
                $options
            );

            $response = $this->http->send($request);
            return [
                'status_code' => $response->getStatusCode(),
                'data' => json_decode($response->getBody()->getContents(), true),
            ];
        } catch (\Throwable $e) {
            return [
                'status_code' => (int) ($e->getCode() ?: 500),
                'error' => $e->getMessage(),
            ];
        }
    }

    public function searchRead(string $model, string $token, array $parameters, ConfigDatabase $server): array
    {
        return $this->authenticatedGet('http://' . $this->apiHost . '/api/search_read/' . $model, $token, $parameters, $server);
    }

    public function read(string $model, string $token, array $parameters, ConfigDatabase $server): array
    {
        return $this->authenticatedGet('http://' . $this->apiHost . '/api/read/' . $model, $token, $parameters, $server);
    }

    public function create(string $model, string $token, array $payload, ConfigDatabase $server): array
    {
        return $this->authenticatedJson('POST', 'http://' . $this->apiHost . '/api/create/' . $model, $token, $payload, $server);
    }

    public function write(string $model, string $token, array $payload, ConfigDatabase $server): array
    {
        return $this->authenticatedJson('PUT', 'http://' . $this->apiHost . '/api/write/' . $model, $token, $payload, $server);
    }

    private function authenticatedGet(string $url, string $token, array $query, ConfigDatabase $server): array
    {
        try {
            $provider = $this->getProvider($server);
            $request = $provider->getAuthenticatedRequest('GET', $url, $token, $query);
            $response = $this->http->send($request);

            return [
                'status_code' => $response->getStatusCode(),
                'data' => json_decode($response->getBody()->getContents()),
            ];
        } catch (\Throwable $e) {
            return [
                'status_code' => (int) ($e->getCode() ?: 500),
                'error' => $e->getMessage(),
            ];
        }
    }

    private function authenticatedJson(string $method, string $url, string $token, array $payload, ConfigDatabase $server): array
    {
        try {
            $provider = $this->getProvider($server);
            $request = $provider->getAuthenticatedRequest($method, $url, $token, [
                'headers' => ['Content-Type' => 'application/json'],
                'body'    => json_encode($payload),
            ]);

            $response = $this->http->send($request);
            return [
                'status_code' => $response->getStatusCode(),
                'data' => json_decode($response->getBody()->getContents(), true),
            ];
        } catch (\Throwable $e) {
            return [
                'status_code' => (int) ($e->getCode() ?: 500),
                'error' => $e->getMessage(),
            ];
        }
    }
}
