<?php

namespace App\Livewire;

use App\Models\Branch;
use App\Models\ConfigDatabase;
use App\Models\Patient;
use App\Models\Product;
use App\Models\SaleOrder;
use App\Models\SaleOrderLine;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Livewire\Component;
use Livewire\WithPagination;

class SaleOrdersCrud extends Component
{
    use WithPagination;

    public string $search = '';
    public string $paymentStatusFilter = '';

    public ?int $editingId = null;

    public ?string $date = null;
    public ?string $number = null;
    public bool $first_time = false;
    public ?int $patient_id = null;
    public string $patientSearch = ''; // Para el término de búsqueda
    public string $selectedPatientName = ''; // Para mostrar el nombre elegido

    public ?int $branch_id = null;

    public ?int $odoo_server = null;

    public string $payment_status = 'created';
    public array $paymentStatusOptions = ['created', 'open', 'paid', 'cancel'];

    public float $subtotal = 0.0;
    public float $discount = 0.0;
    public float $tax = 0.0;
    public float $total = 0.0;
    public float $balance = 0.0;

    public array $lines = [];

    public string $productSearch = '';

    public bool $showLineModal = false;
    public ?int $editingLineIndex = null;

    // Propiedades para Pagos
    public bool $showPaymentModal = false;
    public string $payment_type = 'payment';
    public float $payment_amount = 0.0;
    public string $payment_method = 'cash';
    public string $payment_reference = '';

    public array $paymentTypeOptions = [
        ['value' => 'payment', 'text' => 'Pago'],
        ['value' => 'advance', 'text' => 'Anticipo'],
    ];

    public array $paymentMethodOptions = [
        ['value' => 'cash', 'text' => 'Efectivo'],
        ['value' => 'card', 'text' => 'Tarjeta'],
        ['value' => 'check', 'text' => 'Cheque'],
        ['value' => 'transfer', 'text' => 'Transferencia'],
        ['value' => 'codi', 'text' => 'CODI'],
    ];


    protected $queryString = [
        'search' => ['except' => ''],
        'paymentStatusFilter' => ['except' => ''],
    ];

    public function openPaymentModal(): void
    {
        $this->payment_type = 'payment';
        $this->payment_amount = $this->balance; // Por defecto sugerimos el saldo restante
        $this->payment_method = 'cash';
        $this->payment_reference = '';
        $this->showPaymentModal = true;
    }

    public function savePayment(): void
    {
        $this->validate([
            'payment_type' => ['required', 'in:payment,advance'],
            'payment_amount' => ['required', 'numeric', 'min:0.01'],
            'payment_method' => ['required', 'in:cash,card,check,transfer,codi'],
            'payment_reference' => ['nullable', 'string', 'max:255'],
        ]);

        // Regla: Si es 'payment', no puede exceder el saldo
        if ($this->payment_type === 'payment' && $this->payment_amount > $this->balance) {
            $this->addError('payment_amount', 'El importe no puede ser mayor al saldo pendiente en tipo Pago.');
            return;
        }

        DB::transaction(function () {
            \App\Models\Payment::create([
                'sale_order_id' => $this->editingId,
                'user_id' => Auth::id(),
                'type' => $this->payment_type,
                'amount' => $this->payment_amount,
                'method' => $this->payment_method,
                'reference' => $this->payment_reference ?? '',
                'status' => 'created',
            ]);

            // Si el saldo llega a 0 tras este pago, podrías actualizar el estatus de la orden
            $order = SaleOrder::find($this->editingId);
            if ($order && $order->balance() <= 0) {
                $order->update(['payment_status' => 'paid']);
                $this->payment_status = 'paid';
            }
        });

        $this->showPaymentModal = false;
        $this->recalculateOrderTotalsFromLines(); // Esto refresca el $this->balance
        $this->dispatch('toast', type: 'success', message: '¡Pago registrado con éxito!');
    }

    public function mount(): void
    {
        $this->resetToInitialCreate();
    }

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    public function updatedPaymentStatusFilter(): void
    {
        $this->resetPage();
    }

    public function rules(): array
    {
        return [
            'date' => ['nullable', 'date'],
            'number' => ['nullable', 'string', 'max:255'],

            'first_time' => ['boolean'],
            'patient_id' => ['nullable', 'integer', 'exists:patients,id'],
            'branch_id' => ['nullable', 'integer', 'exists:branches,id'],
            'odoo_server' => ['nullable', 'integer', 'exists:config_databases,id'],


            'lines' => ['array', 'min:0'],
            'lines.*.product_id' => ['nullable', 'integer', 'exists:products,id'],
            'lines.*.quantity' => ['required', 'integer', 'min:1'],
            'lines.*.discount_percentage' => ['nullable', 'numeric', 'min:0', 'max:100'],

            // Calculados
            'lines.*.product_name' => ['nullable', 'string', 'max:255'],
            'lines.*.price' => ['required', 'numeric', 'min:0'],
            'lines.*.discount' => ['nullable', 'numeric', 'min:0'],
            'lines.*.amount' => ['nullable', 'numeric', 'min:0'],
            'lines.*.tax' => ['nullable', 'numeric', 'min:0'],
        ];
    }

    public function resetToInitialCreate(): void
    {
        $this->editingId = null;

        $this->date = now()->toDateString();
        $this->number = null;

        $this->patient_id = null;
        $this->selectedPatientName = '';
        $this->odoo_server = null;
        $this->first_time = false;

        $this->branch_id = Auth::user()->branch_id ?? null;
        $this->payment_status = 'created';

        $this->lines = [];
        $this->subtotal = 0;
        $this->discount = 0;
        $this->tax = 0;
        $this->total = 0;
        $this->balance = 0;

        $this->showLineModal = false;
        $this->editingLineIndex = null;



        $this->resetValidation();
    }

    public function createEmptyOrder(): void
    {
        $this->validate([
            'patient_id' => ['required', 'integer', 'exists:patients,id'],
            'odoo_server' => ['required', 'integer', 'exists:config_databases,id'],
            'first_time' => ['boolean'],
        ]);

        $user = User::query()->findOrFail(Auth::id());

        DB::transaction(function () use ($user) {
            $branch = Branch::query()
                ->select('id', 'sale_order_prefix')
                ->where('id', $user->branch_id)
                ->firstOrFail();

            $order_count = (int) SaleOrder::query()
                    ->where('branch_id', $user->branch_id)
                    ->count() + 1;

            $order_count = str_pad((string) $order_count, 6, '0', STR_PAD_LEFT);
            $folio = $branch->sale_order_prefix . '-' . $order_count;

            $order = SaleOrder::query()->create([
                'date' => now()->toDateString(),
                'number' => $folio,
                'first_time' => (bool) $this->first_time,
                'patient_id' => $this->patient_id,
                'branch_id' => $user->branch_id,

                'subtotal' => 0,
                'discount' => 0,
                'tax' => 0,
                'total' => 0,
                'payment_status' => 'created',

                'odoo_server' => $this->odoo_server,

                'created_by' => $user->id,
                'created_by_name' => $user->name ?? null,
            ]);

            $this->editingId = $order->id;
            $this->date = (string) $order->date;
            $this->number = $order->number;
            $this->branch_id = $order->branch_id;
            $this->payment_status = $order->payment_status;

            $this->lines = [];
            $this->recalculateOrderTotalsFromLines();
        });

        $this->dispatch('toast', type: 'success', message: 'Orden creada. Ya puedes agregar conceptos.');
    }

    public function edit(int $id): void
    {
        $order = SaleOrder::query()
            ->with('lines')
            ->findOrFail($id);

        $this->balance = $order->balance();

        $this->editingId = $order->id;

        $this->date = $order->date ? (string) $order->date : null;
        $this->number = $order->number;

        $this->first_time = (bool) $order->first_time;
        $this->patient_id = $order->patient_id;
        $this->selectedPatientName = $order->patient->name ?? '';
        $this->patientSearch = '';
        $this->branch_id = $order->branch_id;
        $this->odoo_server = $order->odoo_server;

        $this->payment_status = (string) $order->payment_status;

        $this->lines = $order->lines
            ->map(function (SaleOrderLine $l) {
                return [
                    'id' => $l->id,
                    'product_id' => $l->product_id,
                    'product_name' => $l->product_name,
                    'quantity' => (int) $l->quantity,
                    'price' => (float) $l->price,
                    'discount_percentage' => (float) ($l->discount_percentage ?? 0),
                    'discount' => (float) $l->discount,
                    'amount' => (float) $l->amount,
                    'tax' => (float) $l->tax,
                ];
            })
            ->values()
            ->all();

        $this->recalculateOrderTotalsFromLines();




        $this->resetValidation();

        $this->showLineModal = false;
        $this->editingLineIndex = null;
    }

    public function addLine(): void
    {
        $this->lines[] = [
            'id' => null,
            'product_id' => null,
            'product_name' => null,
            'quantity' => 1,
            'price' => 0,
            'discount_percentage' => 0,
            'discount' => 0,
            'amount' => 0,
            'tax' => 0,
        ];
    }

    public function addLineAndOpenModal(): void
    {
        $this->productSearch = '';
        $this->addLine();
        $this->editingLineIndex = count($this->lines) - 1;

        $this->openLineModal($this->editingLineIndex);

        $this->resetValidation();
    }

    public function removeLine(int $index): void
    {
        if (!isset($this->lines[$index])) {
            return;
        }

        array_splice($this->lines, $index, 1);
        $this->recalculateOrderTotalsFromLines();
    }

    public function openLineModal(int $index): void
    {
        if (!isset($this->lines[$index])) {
            return;
        }

        $this->productSearch = '';
        $this->editingLineIndex = $index;
        $this->showLineModal = true;
    }

    public function closeLineModal(): void
    {
        // Si el usuario cerró sin seleccionar producto, NO dejamos la línea vacía.
        if ($this->editingLineIndex !== null && isset($this->lines[$this->editingLineIndex])) {
            $line = $this->lines[$this->editingLineIndex];
            $isNew = empty($line['id']);
            $noProduct = empty($line['product_id']);

            if ($isNew && $noProduct) {
                array_splice($this->lines, $this->editingLineIndex, 1);

            }
        }

        $this->showLineModal = false;
        $this->editingLineIndex = null;

        $this->recalculateOrderTotalsFromLines();
        $this->saveLines();
    }

    public function updatedLines($value = null, $name = null): void
    {
        if (!is_string($name)) {
            return;
        }

        // Detectar cambio en product_id, quantity o discount_percentage
        if (preg_match('/^lines\.(\d+)\.(product_id|quantity|discount_percentage)$/', $name, $m)) {
            $index = (int) $m[1];

            if (!isset($this->lines[$index])) {
                return;
            }

            // Si cambió el producto, cargamos sus datos base primero
            if ($m[2] === 'product_id' && $value) {
                $product = Product::query()
                    ->where('id', $value)
                    ->first();

                if ($product) {
                    $this->lines[$index]['product_name'] = $product->name;
                    $this->lines[$index]['price'] = (float) ($product->list_price ?? 0);
                }
            }

            // Calculamos la línea (Subtotal e IVA)
            $this->recalculateLine($index);

            // Calculamos los totales generales de la orden
            $this->recalculateOrderTotalsFromLines();
        }
    }

    public function updated($propertyName): void
    {
        if (str_starts_with($propertyName, 'lines.')) {
            $parts = explode('.', $propertyName);
            if (isset($parts[1]) && is_numeric($parts[1])) {
                $index = (int) $parts[1];
                $this->recalculateLine($index);
                $this->recalculateOrderTotalsFromLines();
            }
        }
    }

    private function recalculateLine(int $index): void
    {
        if (!isset($this->lines[$index])) {
            return;
        }

        $productId = $this->lines[$index]['product_id'] ?? null;
        $qty = (int) ($this->lines[$index]['quantity'] ?? 1);
        $discountPercentage = (float) ($this->lines[$index]['discount_percentage'] ?? 0);

        $product = null;
        if ($productId) {
            $product = Product::query()
                ->where('id', (int) $productId)
                ->when($this->odoo_server !== null, fn ($q) => $q->where('odoo_server', $this->odoo_server))
                ->first();
        }

        if (!$product) {
            $this->lines[$index]['discount'] = 0;
            $this->lines[$index]['amount'] = 0;
            $this->lines[$index]['tax'] = 0;
            return;
        }

        $price = (float) ($product->list_price ?? 0);
        $this->lines[$index]['price'] = $price;
        $this->lines[$index]['product_name'] = $product->name;

        $discount = 0.0;
        if ($discountPercentage > 0) {
            $discount = $price * ((1 / 100) * $discountPercentage);
        }

        $amount = ($price - $discount) * max(1, $qty);
        $tax = !empty($product->taxes) ? ($amount * 0.16) : 0.0;

        $this->lines[$index]['discount'] = round($discount, 2);
        $this->lines[$index]['amount'] = round($amount, 2);
        $this->lines[$index]['tax'] = round($tax, 2);
    }



    private function recalculateOrderTotalsFromLines(): void
    {
        $discount = 0.0;
        $tax = 0.0;
        $subtotal = 0.0;

        foreach ($this->lines as $line) {
            $discount += (float) ($line['discount'] ?? 0);
            $tax += (float) ($line['tax'] ?? 0);
            $subtotal += (float) ($line['amount'] ?? 0);
        }

        $this->discount = round($discount, 2);
        $this->tax = round($tax, 2);
        $this->subtotal = round($subtotal, 2);
        $this->total = round($this->subtotal + $this->tax, 2);

        if ($this->editingId) {
            $totalPayments = (float) \App\Models\Payment::where('sale_order_id', $this->editingId)->sum('amount');
            $this->balance = round($this->total - $totalPayments, 2);
        }else {
            $this->balance = $this->total;
        }
    }

    public function saveLines(): void
    {
        if (!$this->editingId) {
            $this->dispatch('toast', type: 'error', message: 'Primero crea la orden (Paciente/Servidor/Primera visita).');
            return;
        }

        // Permitir guardar con 0 líneas
        $this->validate([
            'payment_status' => ['required', Rule::in($this->paymentStatusOptions)],
            'lines' => ['array'],
        ]);

        DB::transaction(function () {
            $order = SaleOrder::query()->with('lines')->findOrFail($this->editingId);
            $this->balance = $order->balance();
            // Si no hay líneas, totales en cero y listo
            if (count($this->lines) === 0) {
                $order->update([
                    'discount' => 0,
                    'tax' => 0,
                    'subtotal' => 0,
                    'total' => 0,
                    'payment_status' => $this->payment_status,
                ]);

                SaleOrderLine::query()->where('sale_order_id', $order->id)->delete();

                $this->discount = 0;
                $this->tax = 0;
                $this->subtotal = 0;
                $this->total = 0;

                return;
            }

            // Validación para líneas existentes (solo si hay líneas)
            $this->validate([
                'lines.*.id' => ['nullable', 'integer'],
                'lines.*.product_id' => ['required', 'integer', 'exists:products,id'],
                'lines.*.quantity' => ['required', 'integer', 'min:1'],
                'lines.*.discount_percentage' => ['nullable', 'numeric', 'min:0', 'max:100'],
            ]);

            foreach ($this->lines as $i => $_line) {
                $this->recalculateLine($i);
            }
            $this->recalculateOrderTotalsFromLines();

            $order->update([
                'discount' => $this->discount,
                'tax' => $this->tax,
                'subtotal' => $this->subtotal,
                'total' => $this->total,
                'payment_status' => $this->payment_status,
            ]);

            $existingIds = $order->lines()->pluck('id')->all();
            $sentIds = [];

            foreach ($this->lines as $line) {
                $ok = Product::query()
                    ->where('id', (int) $line['product_id'])
                    ->where('odoo_server', $order->odoo_server)
                    ->exists();

                if (!$ok) {
                    throw new \RuntimeException('Producto no pertenece al servidor de la orden.');
                }

                $saved = SaleOrderLine::query()->updateOrCreate(
                    ['id' => $line['id'] ?? null],
                    [
                        'sale_order_id' => $order->id,
                        'product_id' => (int) $line['product_id'],
                        'product_name' => $line['product_name'],
                        'quantity' => (int) $line['quantity'],
                        'price' => (float) $line['price'],
                        'discount_percentage' => (float) ($line['discount_percentage'] ?? 0),
                        'discount' => (float) $line['discount'],
                        'amount' => (float) $line['amount'],
                        'tax' => (float) $line['tax'],
                    ]
                );

                $sentIds[] = $saved->id;
            }

            $toDelete = array_diff($existingIds, $sentIds);
            if (!empty($toDelete)) {
                SaleOrderLine::query()->whereIn('id', $toDelete)->delete();
            }
        });


        $this->edit($this->editingId);
    }

    public function delete(int $id): void
    {
        $order = SaleOrder::query()->findOrFail($id);
        $order->delete();

        $this->dispatch('toast', type: 'success', message: 'Orden eliminada.');

        if ($this->editingId === $id) {
            $this->resetToInitialCreate();
        }
    }

    public function selectPatient($id, $name): void
    {
        $this->patient_id = $id;
        $this->selectedPatientName = $name;
        $this->patientSearch = ''; // Limpiamos la búsqueda
    }

    public function selectProduct($productId): void
    {
        if ($this->editingLineIndex === null) return;

        $this->lines[$this->editingLineIndex]['product_id'] = $productId;

        // Forzamos la actualización de datos del producto (precio, nombre, etc.)
        $this->updatedLines(null, "lines.{$this->editingLineIndex}.product_id");

        $this->productSearch = ''; // Limpiamos búsqueda tras seleccionar
    }

    public function printTicket(int $id): void
    {
        // Emitimos un evento de JS para abrir el ticket en una pestaña nueva
        $this->dispatch('open-ticket', ['url' => route('sale-orders.ticket', $id)]);
    }

    public function render()
    {
        $orders = SaleOrder::query()
            ->with('patient')
            ->when($this->paymentStatusFilter !== '', fn ($q) => $q->where('payment_status', $this->paymentStatusFilter))
            ->when($this->search !== '', function ($q) {
                $q->where(function ($qq) {
                    $qq->where('number', 'like', '%' . $this->search . '%')
                        ->orWhereHas('patient', fn ($qp) => $qp->where('name', 'like', '%' . $this->search . '%'));
                });
            })
            ->orderByDesc('id')
            ->paginate(10);

        $patients = Patient::query()
            ->when($this->patientSearch !== '', function ($q) {
                $q->where('name', 'like', '%' . $this->patientSearch . '%');
            })
            ->orderBy('name')
            ->limit(10)
            ->get(['id', 'name']);
        $servers = ConfigDatabase::query()->orderBy('database')->get(['id', 'database']);

        $products = Product::query()
            ->where('active', 1)
            ->when($this->odoo_server !== null, fn ($q) => $q->where('odoo_server', $this->odoo_server))
            ->when($this->productSearch !== '', fn ($q) => $q->where('name', 'like', '%' . $this->productSearch . '%'))
            ->orderBy('name')
            ->limit(10) // Limitamos para que el modal no sea gigante
            ->get(['id', 'name', 'list_price', 'taxes', 'odoo_server']);

        return view('livewire.sale-orders-crud', [
            'orders' => $orders,
            'patients' => $patients,
            'servers' => $servers,
            'products' => $products,
        ]);
    }
}
