<?php

namespace App\Livewire\Reports\Sales;

use App\Models\SaleOrder;
use App\Models\SaleOrderLine;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Livewire\WithPagination;

class SalesReport extends Component
{
    use WithPagination;



    public $fromDate;
    public $toDate;
    public $reportType = 'daily_concept';
    public $userId;
    public $usersList = [];
    public $readyToLoad = false; // Para no cargar datos al entrar sin darle click

    protected $queryString = [
        'fromDate',
        'toDate',
        'reportType',
        'userId',
        'page' => ['except' => 1]
    ];

    public function mount()
    {
        $this->fromDate = $this->fromDate ?? now()->startOfMonth()->format('Y-m-d');
        $this->toDate = $this->toDate ?? now()->format('Y-m-d');
        $this->usersList = User::forDropDown(true);
    }

    public function updated($propertyName)
    {
        if (in_array($propertyName, ['fromDate', 'toDate', 'reportType', 'userId'])) {
            $this->resetPage();
            $this->readyToLoad = false;
        }
    }

    public function generateReport()
    {
        $rules = [
            'fromDate' => 'required|date',
            'toDate' => 'required|date|after_or_equal:fromDate',
            'reportType' => 'required',
        ];

        if ($this->reportType === 'user_detailed') {
            $rules['userId'] = 'required';
        }

        $this->validate($rules);
        $this->readyToLoad = true;
    }

    private function getDailyByConceptQuery()
    {
        return SaleOrderLine::whereHas('saleOrder', function ($query) {
            $query->whereBetween('date', [$this->fromDate, $this->toDate]);
        })
            ->select('product_name', DB::raw('SUM(quantity) as quantity'), 'price', DB::raw('SUM(discount) as discount'), DB::raw('SUM(amount) as amount'))
            ->groupBy('product_name', 'price');
    }

    private function getUserDetailedQuery()
    {
        $query = SaleOrder::with(['patient', 'user', 'lines'])
            ->whereBetween('date', [$this->fromDate, $this->toDate]);

        if ($this->userId != 0) {
            $query->where('created_by', $this->userId);
        }

        return $query->orderBy('date', 'asc');
    }

    private function getDailySalQuery()
    {
        return SaleOrderLine::with(['saleOrder.payments'])
            ->whereHas('saleOrder', function ($query) {
                $query->whereBetween('date', [$this->fromDate, $this->toDate]);
            });
    }

    public function exportToExcel()
    {
        $this->generateReport();

        // Obtenemos todos los datos sin paginar para el Excel
        $results = [];
        if ($this->reportType === 'daily_concept') {
            $results = $this->getDailyByConceptQuery()->get()->toArray();
        } elseif ($this->reportType === 'user_detailed') {
            $results = $this->transformUserDetailed($this->getUserDetailedQuery()->get());
        } else {
            $results = $this->transformDailySal($this->getDailySalQuery()->get());
        }

        $filename = "reporte_{$this->reportType}_{$this->fromDate}_al_{$this->toDate}.csv";

        return response()->streamDownload(function () use ($results) {
            $handle = fopen('php://output', 'w');
            fputs($handle, chr(0xEF).chr(0xBB).chr(0xBF));
            if (!empty($results)) {
                fputcsv($handle, array_keys($results[0]));
                foreach ($results as $row) { fputcsv($handle, $row); }
            }
            fclose($handle);
        }, $filename);
    }

    // Transformadores para mantener la vista limpia
    private function transformUserDetailed($orders) {
        $data = [];
        foreach ($orders as $order) {
            foreach ($order->lines as $line) {
                $data[] = [
                    'date' => $order->date, 'patient' => $order->patient->name ?? 'N/A',
                    'first_time' => $order->first_time ? 'SI' : 'NO', 'number' => $order->number,
                    'product_name' => $line->product_name, 'quantity' => $line->quantity,
                    'price' => $line->price, 'discount' => $line->discount,
                    'amount' => $line->amount - $line->tax, 'tax' => $line->tax, 'total' => $line->amount
                ];
            }
        }
        return $data;
    }

    private function transformDailySal($lines) {
        $data = [];
        foreach ($lines as $line) {
            $data[] = [
                'product_name' => $line->product_name, 'quantity' => $line->quantity,
                'price' => $line->price, 'discount' => $line->discount, 'tax' => $line->tax,
                'amount' => $line->amount, 'date' => $line->saleOrder->date,
                'number' => $line->saleOrder->number, 'total' => $line->saleOrder->total,
                'paid' => $line->saleOrder->paid(), 'balance' => $line->saleOrder->balance()
            ];
        }
        return $data;
    }

    public function render()
    {
        $results = [];
        if ($this->readyToLoad) {
            if ($this->reportType === 'daily_concept') {
                $results = $this->getDailyByConceptQuery()->paginate(20);
            } elseif ($this->reportType === 'user_detailed') {
                // Para el detallado paginamos las órdenes, aunque las líneas sean más
                $paginatedOrders = $this->getUserDetailedQuery()->paginate(15);
                $transformed = $this->transformUserDetailed($paginatedOrders->getCollection());
                $results = new \Illuminate\Pagination\LengthAwarePaginator(
                    $transformed, $paginatedOrders->total(), $paginatedOrders->perPage(), $paginatedOrders->currentPage(),
                    ['path' => \Illuminate\Pagination\Paginator::resolveCurrentPath()]
                );
            } else {
                $paginatedLines = $this->getDailySalQuery()->paginate(20);
                $transformed = $this->transformDailySal($paginatedLines->getCollection());
                $results = new \Illuminate\Pagination\LengthAwarePaginator(
                    $transformed, $paginatedLines->total(), $paginatedLines->perPage(), $paginatedLines->currentPage(),
                    ['path' => \Illuminate\Pagination\Paginator::resolveCurrentPath()]
                );
            }
        }

        return view('livewire.reports.Sales.sales-report', [
            'results' => $results
        ]);
    }
}
