<?php

namespace App\Livewire;

use App\Models\Patient;
use Livewire\Component;
use Livewire\WithPagination;

class PatientsCrud extends Component
{
    use WithPagination;

    public string $search = '';

    public ?int $editingId = null;

    public ?string $name = null;
    public ?string $birthday = null;
    public string $gender = 'HOMBRE';

    public ?string $company_name = null;
    public string $tin = 'XEXX010101000';

    public ?string $street = null;
    public ?string $street2 = null;
    public ?string $city = null;
    public ?string $state = null;
    public ?string $country = null;

    public ?string $phone = null;
    public ?string $mobile = null;
    public ?string $email = null;

    public array $genderOptions = ['HOMBRE', 'MUJER'];

    protected $queryString = [
        'search' => ['except' => ''],
    ];

    public function mount(): void
    {
        $this->create();
    }

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    public function rules(): array
    {
        return [
            'name' => ['nullable', 'string', 'max:255'],
            'birthday' => ['nullable', 'date'],
            'gender' => ['required', 'in:HOMBRE,MUJER'],

            'company_name' => ['nullable', 'string', 'max:255'],
            'tin' => ['required', 'string', 'max:13'],

            'street' => ['nullable', 'string', 'max:255'],
            'street2' => ['nullable', 'string', 'max:255'],
            'city' => ['nullable', 'string', 'max:255'],
            'state' => ['nullable', 'string', 'max:255'],
            'country' => ['nullable', 'string', 'max:255'],

            'phone' => ['nullable', 'string', 'max:255'],
            'mobile' => ['nullable', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255'],
        ];
    }

    public function create(): void
    {
        $this->editingId = null;

        $this->name = null;
        $this->birthday = null;

        $this->gender = 'HOMBRE';

        $this->company_name = null;
        $this->tin = 'XEXX010101000';

        $this->street = null;
        $this->street2 = null;
        $this->city = null;
        $this->state = null;
        $this->country = 'México';

        $this->phone = null;
        $this->mobile = null;
        $this->email = null;

        $this->resetValidation();
    }

    public function edit(int $id): void
    {
        $p = Patient::query()->findOrFail($id);

        $this->editingId = $p->id;

        $this->name = $p->name;
        $this->birthday = $p->birthday ? (string) $p->birthday : null;

        $this->gender = $p->gender ?: 'HOMBRE';

        $this->company_name = $p->company_name;
        $this->tin = $p->tin ?: 'XEXX010101000';

        $this->street = $p->street;
        $this->street2 = $p->street2;
        $this->city = $p->city;
        $this->state = $p->state;
        $this->country = $p->country;

        $this->phone = $p->phone;
        $this->mobile = $p->mobile;
        $this->email = $p->email;

        $this->resetValidation();
    }

    public function save(): void
    {
        $data = $this->validate();

        Patient::query()->updateOrCreate(
            ['id' => $this->editingId],
            $data
        );

        $this->dispatch('toast', type: 'success', message: $this->editingId ? 'Paciente actualizado.' : 'Paciente creado.');
        $this->create();
    }

    public function delete(int $id): void
    {
        $p = Patient::query()->findOrFail($id);
        $p->delete();

        $this->dispatch('toast', type: 'success', message: 'Paciente eliminado.');

        if ($this->editingId === $id) {
            $this->create();
        }
    }

    public function render()
    {
        $patients = Patient::query()
            ->when($this->search !== '', function ($q) {
                $q->where(function ($qq) {
                    $qq->where('name', 'like', '%' . $this->search . '%')
                        ->orWhere('email', 'like', '%' . $this->search . '%')
                        ->orWhere('mobile', 'like', '%' . $this->search . '%')
                        ->orWhere('phone', 'like', '%' . $this->search . '%')
                        ->orWhere('tin', 'like', '%' . $this->search . '%');
                });
            })
            ->orderBy('name')
            ->paginate(10);

        return view('livewire.patients-crud', [
            'patients' => $patients,
        ]);
    }
}
