<?php

namespace App\Livewire;

use App\Models\Appointment;
use App\Models\AppointmentCategory;
use App\Models\AppointmentClasses;
use App\Models\AppointmentShapes;
use App\Models\AppointmentType;
use App\Models\Branch;
use App\Models\Patient;
use App\Models\SaleOrder;
use App\Models\SaleOrderLine;
use App\Models\User;
use Auth;
use DB;
use Illuminate\Validation\Rule;
use Livewire\Component;
use Livewire\WithPagination;

class AppointmentsCrud extends Component
{
    use WithPagination;

    public string $search = '';
    public string $statusFilter = '';
    public ?string $dateFilter = null;

    public ?int $editingId = null;

    public string $date = '';
    public string $time = '';
    public ?int $patient_id = null;
    public string $patientSearch = ''; // Para el término de búsqueda
    public string $selectedPatientName = ''; // Para mostrar el nombre elegido
    public ?int $branch_id = null;
    public ?int $appointment_type_id = null;

    public ?bool $first_time = false;

    public string $status = 'Creada';
    public ?string $note = null;
    public ?string $prescription_note = null;

    public array $statusOptions = ['Creada','Agendada','En Proceso','Finalizada','Documentada','Cancelada'];

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => ''],
        'dateFilter' => ['except' => null],
    ];

    public function mount(): void
    {
        $this->create();
    }

    public function updatedSearch(): void
    {
        $this->resetPage();
    }

    public function updatedStatusFilter(): void
    {
        $this->resetPage();
    }

    public function updatedDateFilter(): void
    {
        $this->resetPage();
    }

    public function rules(): array
    {
        return [
            'date' => ['required', 'date'],
            'time' => ['required', 'date_format:H:i'],
            'patient_id' => ['required', 'integer', 'exists:patients,id'],
            'branch_id' => ['required', 'integer', 'exists:branches,id'],
            'appointment_type_id' => ['required', 'integer', 'exists:appointment_types,id'],
            'first_time' => ['boolean'],
            'note' => ['nullable', 'string', 'max:255'],

        ];
    }

    public function create(): void
    {
        $this->editingId = null;
        $this->patientSearch = '';
        $this->selectedPatientName = '';

        $this->date = now()->toDateString();
        $this->time = now()->format('H:i');

        $this->patient_id = null;
        $this->branch_id = auth()->user()->branch_id;
        $this->appointment_type_id = null;
        $this->first_time = false;


        $this->status = 'Creada';
        $this->note = null;
        $this->prescription_note = null;

        $this->resetValidation();
    }

    public function edit(int $id): void
    {
        $a = Appointment::query()->findOrFail($id);

        $this->editingId = $a->id;
        $this->patient_id = $a->patient_id;
        $this->selectedPatientName = $a->patient->name ?? '';
        $this->patientSearch = '';
        $this->date = (string) $a->date;
        $this->time = substr((string) $a->time, 0, 5);

        $this->branch_id = $a->branch_id;
        $this->appointment_type_id = $a->appointment_type_id;
        $this->first_time = $a->first_time;

        $this->status = (string) $a->status;
        $this->note = $a->note;
        $this->prescription_note = $a->prescription_note;

        $this->resetValidation();
    }

    public function save(): void
    {
        $data = $this->validate();

        $appointment = Appointment::query()->updateOrCreate(
            ['id' => $this->editingId],
            $data
        );

        if (!$this->editingId) {
            $concepts = $appointment->type->concepts()
                ->where('enabled', true)
                ->with('appointmentConcept')
                ->orderBy('order')
                ->get();

            foreach ($concepts as $concept) {
                $appointment->stages()->create([
                    'appointment_type_concept_id' => $concept->id,
                    'area_id' => $concept->appointmentConcept->area_id,
                    'concept_name' => $concept->appointmentConcept->name,
                    'order' => $concept->order,
                ]);
            }
        }

        $this->dispatch('toast', type: 'success', message: $this->editingId ? 'Cita actualizada.' : 'Cita creada.');
        $this->create();
    }

    public function CreateSaleOrderFromAppointment(int $id): void
    {
        $appointment = Appointment::query()->with(['type.product', 'patient'])->findOrFail($id);

        if ($appointment->sale_order_id) {
            $this->dispatch('toast', type: 'error', message: 'Esta cita ya tiene un ticket generado.');
            return;
        }

        $product = $appointment->type->product;
        if (!$product) {
            $this->dispatch('toast', type: 'error', message: 'El tipo de cita no tiene un producto relacionado.');
            return;
        }

        $user = User::query()->findOrFail(Auth::id());

        DB::transaction(function () use ($appointment, $product, $user) {
            // Generar Folio
            $branch = Branch::query()
                ->select('id', 'sale_order_prefix')
                ->where('id', $appointment->branch_id)
                ->firstOrFail();

            $order_count = (int) SaleOrder::query()
                    ->where('branch_id', $appointment->branch_id)
                    ->count() + 1;

            $order_count = str_pad((string) $order_count, 6, '0', STR_PAD_LEFT);
            $folio = $branch->sale_order_prefix . '-' . $order_count;

            // Cálculos de precio y descuento
            $price = $product->list_price;
            $discountPercent = $appointment->discount_value ?? 0;
            $discountAmount = $price * ($discountPercent / 100);
            $total = $price - $discountAmount;

            // Crear SaleOrder
            $order = SaleOrder::query()->create([
                'date' => now()->toDateString(),
                'number' => $folio,
                'patient_id' => $appointment->patient_id,
                'branch_id' => $appointment->branch_id,
                'subtotal' => $price,
                'discount' => $discountAmount,
                'tax' => 0,
                'total' => $total,
                'payment_status' => 'created',
                'odoo_server' => $product->odoo_server,
                'created_by' => $user->id,
                'created_by_name' => $user->name,
                'first_time' => $appointment->first_time,
            ]);

            // Crear SaleOrderLine
            SaleOrderLine::query()->create([
                'sale_order_id' => $order->id,
                'product_id' => $product->id,
                'product_name' => $product->name,
                'quantity' => 1,
                'price' => $price,
                'discount_percentage' => $discountPercent,
                'discount' => $discountAmount,
                'tax' => 0,
                'amount' => $total,
            ]);

            // Vincular cita con la orden
            $appointment->update(['sale_order_id' => $order->id]);
        });

        $this->dispatch('toast', type: 'success', message: 'Ticket generado correctamente.');
    }

    public function delete(int $id): void
    {
        $a = Appointment::query()->findOrFail($id);
        $a->delete();

        $this->dispatch('toast', type: 'success', message: 'Cita eliminada.');

        if ($this->editingId === $id) {
            $this->create();
        }
    }

    public function selectPatient($id, $name): void
    {
        $this->patient_id = $id;
        $this->selectedPatientName = $name;
        $this->patientSearch = ''; // Limpiamos la búsqueda
    }

    public function render()
    {
        $appointments = Appointment::query()
            ->with(['patient', 'type'])
            ->when($this->statusFilter !== '', fn ($q) => $q->where('status', $this->statusFilter))
            ->when($this->dateFilter, fn ($q) => $q->whereDate('date', $this->dateFilter))
            ->when($this->search !== '', function ($q) {
                $q->whereHas('patient', function ($qp) {
                    // Ajusta el campo si tu paciente no tiene "name"
                    $qp->where('name', 'like', '%' . $this->search . '%');
                });
            })
            ->orderByDesc('date')
            ->orderByDesc('time')
            ->paginate(10);

        $patients = Patient::query()
            ->when($this->patientSearch !== '', function($q) {
                $q->where('name', 'like', '%' . $this->patientSearch . '%');
            })
            ->orderBy('name')
            ->limit(10) // Limitamos a 10 para que sea rápido y manejable
            ->get(['id', 'name']);

        $branches = Branch::query()
            ->orderBy('name')
            ->get(['id', 'name']);

        $types = AppointmentType::query()
            ->orderBy('name')
            ->get(['id', 'name']);

        $categories = class_exists(AppointmentCategory::class)
            ? AppointmentCategory::query()->orderBy('name')->get(['id', 'name'])
            : collect();

        $classes = class_exists(AppointmentClasses::class)
            ? AppointmentClasses::query()->orderBy('name')->get(['id', 'name'])
            : collect();

        $shapes = class_exists(AppointmentShapes::class)
            ? AppointmentShapes::query()->orderBy('name')->get(['id', 'name'])
            : collect();

        return view('livewire.appointments-crud', [
            'appointments' => $appointments,
            'patients' => $patients,
            'branches' => $branches,
            'types' => $types,
            'categories' => $categories,
            'classes' => $classes,
            'shapes' => $shapes,
        ]);
    }
}
