<?php

namespace App\Livewire;

use App\Models\AppointmentFollowUp;
use Livewire\Component;
use App\Models\AppointmentStage;
use App\Models\Appointment;
use App\Models\Area;
use App\Models\SystemDiscount;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class AppointmentTracker extends Component
{
    public $selectedAreaId;
    public $filterDate;

    // Propiedades para el Modal
    public $showFinishModal = false;
    public $activeStageId;
    public $stageObservations;
    public $selectedDiscount = 0;

    public $enableFollowUp = false;
    public $followUpOption = '7 days';
    public $followUpReason = '';

    protected $followUpOptions = [
        '7 days' => '7 Días',
        '15 days' => '15 Días',
        '21 days' => '21 Días',
        '1 month' => '1 Mes',
        '2 months' => '2 Meses',
        '3 months' => '3 Meses',
        '6 months' => '6 Meses',
        '1 year' => '1 Año',
    ];

    public function mount()
    {
        $this->selectedAreaId = Area::first()?->id;
        $this->filterDate = now()->toDateString();
    }

    public function render()
    {
        $areas = Area::all();
        $discounts = SystemDiscount::all();

        $query = AppointmentStage::with(['appointment.patient', 'user', 'appointment.stages'])
            ->where('area_id', $this->selectedAreaId)
            ->whereHas('appointment', fn($q) => $q->whereDate('date', $this->filterDate))
            ->orderBy('appointment_id')
            ->orderBy('order');

        $allStages = $query->get()->map(function($stage) {
            $stage->wait_time = $this->calculateWaitTime($stage);
            return $stage;
        });

        $pendingStages = $allStages->filter(fn($s) => is_null($s->started_at));
        $inProgressStages = $allStages->filter(fn($s) => !is_null($s->started_at) && is_null($s->ended_at));

        return view('livewire.appointment-tracker', compact('areas', 'pendingStages', 'inProgressStages', 'discounts'));
    }

    private function calculateWaitTime($stage)
    {
        // Si ya inició, no hay tiempo de espera actual
        if ($stage->started_at) return 0;

        $appointment = $stage->appointment;
        $now = now();
        // Si es el primer stage (order 1), comparar con la hora de la cita
        if ($stage->order === 1) {
            $appointmentTime = Carbon::parse($appointment->date->format('Y-m-d') . ' ' . $appointment->time);

            return $now->greaterThan($appointmentTime)
                ? floor($appointmentTime->diffInMinutes($now))
                : 0;
        }

        // Si no es el primero, comparar con el fin del stage anterior
        $previousStage = $appointment->stages->where('order', $stage->order - 1)->first();
        if ($previousStage && $previousStage->ended_at) {
            return $now->greaterThan($previousStage->ended_at)
                ? floor($previousStage->ended_at->diffInMinutes($now))
                : 0;
        }

        return 0;
    }

    public function startStage($stageId)
    {
        // Restricción de fecha futura
        if (Carbon::parse($this->filterDate)->isFuture()) {
            session()->flash('error', 'No se pueden iniciar etapas de citas futuras.');
            return;
        }

        $stage = AppointmentStage::findOrFail($stageId);

        // Evitar que otro usuario lo inicie si ya alguien lo hizo (Update status manual)
        $stage->refresh();
        if ($stage->started_at) {
            session()->flash('error', 'Esta etapa ya ha sido iniciada por otro usuario.');
            return;
        }

        if (AppointmentStage::REQUIRE_PREVIOUS_FINISHED && $stage->order > 1) {
            $prev = AppointmentStage::where('appointment_id', $stage->appointment_id)
                ->where('order', '<', $stage->order)
                ->whereNull('ended_at')->exists();
            if ($prev) {
                session()->flash('error', 'Etapas anteriores pendientes.');
                return;
            }
        }

        $now = now();
        $stage->update(['started_at' => $now, 'user_id' => Auth::id()]);

        // Si es el primer stage, marcar inicio de la Cita
        if ($stage->order === 1) {
            $stage->appointment->update(['started_at' => $now, 'status' => 'En Proceso']);
        }
    }

    public function openFinishModal($stageId)
    {
        $this->activeStageId = $stageId;
        $this->showFinishModal = true;
        $this->stageObservations = '';
        $this->selectedDiscount = 0;
    }

    public function endStage()
    {
        $now = now();

        $stage = AppointmentStage::findOrFail($this->activeStageId);

        if ($stage->user_id !== Auth::id()) {
            session()->flash('error', 'No tienes permiso para finalizar esta etapa.');
            return;
        }


        // Verificar si es el último stage para cerrar la cita
        $isLast = !AppointmentStage::where('appointment_id', $stage->appointment_id)
            ->where('order', '>', $stage->order)->exists();

        if ($isLast) {
            $appointment = $stage->appointment;
            $totalDuration = $appointment->started_at ? $appointment->started_at->diffInMinutes($now) : 0;
            $appointment->update([
                'ended_at' => $now,
                'total_duration_minutes' => $totalDuration,
                'status' => 'Finalizada',
                'discount_value' => $this->selectedDiscount
            ]);

            if ($this->enableFollowUp) {

                $this->validate([
                    'followUpReason' => 'required|string|min:3',
                ], [
                    'followUpReason.required' => 'El motivo del seguimiento es obligatorio si la opción está marcada.',
                ]);

                AppointmentFollowUp::create([
                    'patient_id' => $appointment->patient_id,
                    'appointment_id' => $appointment->id,
                    'due_date' => now()->add($this->followUpOption),
                    'reason' => $this->followUpReason,
                    'status' => 'open'
                ]);
            }

        }


        $stage->update([
            'ended_at' => $now,
            'duration_minutes' => $stage->started_at->diffInMinutes($now),
            'observations' => $this->stageObservations
        ]);


        $this->showFinishModal = false;
        $this->reset(['activeStageId', 'stageObservations', 'selectedDiscount', 'enableFollowUp', 'followUpReason']);
    }


    public function refreshData() { /* Solo para disparar el re-render */ }
}
