<?php

namespace App\Filament\Resources;

use App\Filament\Resources\UserResource\Pages;
use App\Models\Branch;
use App\Models\User;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select as FormSelect;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;

class UserResource extends Resource
{
    protected static ?string $model = User::class;
    protected static ?string $slug = 'usuarios';
    protected static ?string $navigationIcon = 'heroicon-o-user';
    protected static ?string $navigationGroup = 'Usuarios';
    protected static ?string $navigationLabel = 'Usuarios';
    protected static ?string $pluralModelLabel = 'Usuarios';
    protected static ?string $modelLabel = 'Usuario';

    public static function form(Form $form): Form
    {
        // Preload options outside closures to avoid null components in schema evaluation.
        $roleOptions = Role::query()->pluck('name', 'name')->all();
        $branchOptions = Branch::query()->orderBy('name')->pluck('name', 'id')->all();

        return $form
            ->schema([
                Section::make('Datos del Usuario')
                    ->schema([
                        Grid::make(2)->schema([
                            TextInput::make('name')
                                ->label('Nombre')
                                ->required()
                                ->maxLength(255),
                            TextInput::make('email')
                                ->label('Email')
                                ->email()
                                ->required()
                                ->unique(ignoreRecord: true),
                        ]),
                        Grid::make(3)->schema([
                            TextInput::make('password')
                                ->label('Contraseña')
                                ->password()
                                ->revealable()
                                ->required(fn (string $operation): bool => $operation === 'create')
                                ->minLength(8)
                                ->dehydrated(fn ($state) => filled($state))
                                ->dehydrateStateUsing(fn ($state) => filled($state) ? Hash::make($state) : null),
                            TextInput::make('odoo_password')
                                ->label('Contraseña Odoo')
                                ->password()
                                ->revealable()
                                ->dehydrated(fn ($state) => filled($state)),
                            FormSelect::make('branch_id')
                                ->label('Sucursal')
                                ->options($branchOptions)
                                ->searchable()
                                ->preload()
                                ->required(),
                        ]),

                    ]),

                Section::make('Permisos')
                    ->schema([
                        FormSelect::make('roles')
                            ->label('Rol')
                            ->preload()
                            ->searchable()
                            ->options($roleOptions)
                            ->default([])
                            ->dehydrated(true),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            // Avoid type-hint conflicts; simply accept $query and return it.
            ->modifyQueryUsing(function ($query) {
                return $query->with(['roles', 'branch']);
            })
            ->columns([
                TextColumn::make('name')->label('Nombre')->sortable()->searchable(),
                TextColumn::make('email')->label('Email')->sortable()->searchable(),
                TextColumn::make('branch.name')->label('Sucursal')->sortable()->searchable(),
                TextColumn::make('roles_list')
                    ->label('Roles')
                    ->getStateUsing(fn (User $record) => ($record->roles ?? collect())->pluck('name')->implode(', ')),
                IconColumn::make('is_active')->label('Activo')->boolean(),
                TextColumn::make('created_at')->label('Creado')->dateTime()->since()->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([])
            ->actions([
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }

    public static function getGloballySearchableAttributes(): array
    {
        return ['name', 'email'];
    }
}
