<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ProductResource\Pages;
use App\Models\ConfigDatabase;
use App\Models\Product;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select as FormSelect;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;

class ProductResource extends Resource
{
    protected static ?string $model = Product::class;

    protected static ?string $slug = 'products';

    protected static ?string $navigationIcon = 'heroicon-o-cube';
    protected static ?string $navigationGroup = 'Catálogos';
    protected static ?string $navigationLabel = 'Productos';
    protected static ?string $pluralModelLabel = 'Productos';
    protected static ?string $modelLabel = 'Producto';

    public static function form(Form $form): Form
    {
        $serverOptions = ConfigDatabase::query()->orderBy('host')->pluck('host', 'id')->all();

        return $form
            ->schema([
                Section::make('Información del producto')
                    ->schema([
                        Grid::make(2)->schema([
                            TextInput::make('name')
                                ->label('Nombre')
                                ->required()
                                ->maxLength(191),
                            TextInput::make('list_price')
                                ->label('Precio de lista')
                                ->numeric()
                                ->minValue(0),
                        ]),
                        Grid::make(3)->schema([
                            TextInput::make('odoo_product_id')
                                ->label('ID en Odoo')
                                ->numeric()
                                ->disabled()
                                ->minValue(1),
                            FormSelect::make('odoo_server')
                                ->label('Servidor Odoo')
                                ->options($serverOptions) // value=id, label=host
                                ->searchable()
                                ->disabled()
                                ->preload()
                                ->native(false),
                            TextInput::make('business_unit_id')
                                ->label('Unidad de negocio ID')
                                ->disabled()
                                ->numeric()
                                ->minValue(1),
                        ]),
                        Toggle::make('taxes')
                            ->label('Impuestos')
                            ->default(true),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(function ($query) {
                return $query->with('server');
            })
            ->columns([
                TextColumn::make('name')->label('Nombre')->sortable()->searchable(),
                TextColumn::make('list_price')->label('Precio')->money()->sortable(),
                TextColumn::make('server.database')->label('Base de Datos')->sortable()->searchable(),
                TextColumn::make('odoo_product_id')->label('Odoo ID')->sortable(),
                TextColumn::make('business_unit_id')->label('Unidad de negocio')->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('taxes')->label('Impuestos')->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('created_at')->label('Creado')->dateTime()->since()->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([])
            ->actions([
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListProducts::route('/'),
            'create' => Pages\CreateProduct::route('/create'),
            'edit'   => Pages\EditProduct::route('/{record}/edit'),
        ];
    }

    public static function getGloballySearchableAttributes(): array
    {
        return ['name', 'odoo_product_id'];
    }
}
