<?php

namespace App\Filament\Resources;

use App\Filament\Resources\AppointmentTypeResource\Pages;
use App\Models\AppointmentConcept;
use App\Models\AppointmentType;
use App\Models\Area;
use App\Models\Product;
use Filament\Forms\Components\Grid;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select as FormSelect;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;

class AppointmentTypeResource extends Resource
{
    protected static ?string $model = AppointmentType::class;

    protected static ?string $slug = 'appointment-types';
    protected static ?string $navigationIcon = 'heroicon-o-list-bullet';
    protected static ?string $navigationGroup = 'Configuración';
    protected static ?string $navigationLabel = 'Tipos de Cita';
    protected static ?string $pluralModelLabel = 'Tipos de Cita';
    protected static ?string $modelLabel = 'Tipo de Cita';

    public static function form(Form $form): Form
    {
        // Select shows Product.name as label, saves id as value
        $productOptions = Product::query()->orderBy('name')->pluck('name', 'id')->all();
        $conceptOptions = AppointmentConcept::query()->orderBy('name')->pluck('name', 'id')->all();
        return $form
            ->schema([
                Section::make('Información')
                    ->schema([
                        Grid::make(6)->schema([
                            TextInput::make('name')
                                ->label('Nombre')
                                ->required()
                                ->columnSpan(2)
                                ->maxLength(191),
                            FormSelect::make('related_product_id')
                                ->label('Producto relacionado')
                                ->options($productOptions) // label=name, value=id
                                ->searchable()
                                ->preload()
                                ->columnSpan(3)
                                ->native(false),
                            TextInput::make('quota')
                                ->label('Cuota Mensual (CANT)')
                                ->numeric()
                        ]),
                        Toggle::make('active')
                            ->label('Activo')
                            ->default(true),
                    ]),
                Section::make('Pasos')
                    ->schema([
                        Repeater::make('concepts')
                            ->label('Pasos del tipo de cita')
                            ->relationship('concepts') // uses hasMany(AppointmentTypeStep::class)
                            ->defaultItems(0)
                            ->orderColumn('order')
                            ->reorderableWithButtons()
                            ->schema([
                                Grid::make(5)->schema([

                                    FormSelect::make('appointment_concept_id')
                                        ->label('Concepto')
                                        ->options($conceptOptions) // label=name, value=id
                                        ->searchable()
                                        ->preload()
                                        ->required()
                                        ->columnSpan(2)
                                        ->native(false),

                                    TextInput::make('expected_time')
                                        ->label('T esperado (min)')
                                        ->numeric()
                                        ->minValue(0),

                                    TextInput::make('maximum_time')
                                        ->label('T Maximo (min)')
                                        ->numeric()
                                        ->minValue(0),

                                    TextInput::make('order')
                                        ->hidden()
                                        ->label('Orden')
                                        ->numeric()
                                        ->minValue(0),
                                    Toggle::make('enabled')
                                        ->label('Activo')
                                        ->default(true),
                                ]),

                            ])

                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(function ($query) {
                return $query->with(['product']);
            })
            ->columns([
                TextColumn::make('name')->label('Nombre')->sortable()->searchable(),
                TextColumn::make('product.name')->label('Producto')->sortable()->searchable(),
                IconColumn::make('active')->label('Activo')->boolean(),
                TextColumn::make('created_at')->label('Creado')->dateTime()->since()->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([])
            ->actions([
                EditAction::make(),
                DeleteAction::make(),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListAppointmentTypes::route('/'),
            'create' => Pages\CreateAppointmentType::route('/create'),
            'edit'   => Pages\EditAppointmentType::route('/{record}/edit'),
        ];
    }

    public static function getGloballySearchableAttributes(): array
    {
        return ['name'];
    }
}
